#include	"h_MMC3.h"

namespace MMC3 {
MMC3Type	type;
MMC3Revision	revision;
uint8_t		pointer;
uint8_t		reg[8];
uint8_t		mirroring;
uint8_t		wram;
bool		wramAlwaysEnabled;

uint8_t		counter;
uint8_t		prescaler;
uint8_t		reloadValue;
bool		reload;
bool		enableIRQ;
uint8_t		pa12Filter;
uint16_t	prevAddr;

FSync		sync;
FCPURead	wramRead, wramReadCallback;
FCPUWrite	wramWrite, wramWriteCallback;

int		(*GetPRGBank)(int);
int		(*GetCHRBank)(int);


void	syncWRAM (void) {
	if (type ==MMC3Type::MMC6) {
		if (pointer &0x20)
			EMU->SetPRG_RAM4(0x7, 0);
		else
			EMU->SetPRG_OB4(0x7);
	} else
	if (wram &0x80 || wramAlwaysEnabled) {
		EMU->SetPRG_RAM8(0x6, 0);
		if (wram &0x40) {
			EMU->SetPRG_Ptr4(0x6, EMU->GetPRG_Ptr4(0x6), FALSE);
			EMU->SetPRG_Ptr4(0x7, EMU->GetPRG_Ptr4(0x7), FALSE);
		}
	} else {
		EMU->SetPRG_OB4(0x6);
		EMU->SetPRG_OB4(0x7);
	}
}

int	getPRGBank (int bank) {
	if (pointer &0x40 && ~bank &1) bank ^=2;
	return bank &2? 0xFE | bank &1: reg[6 | bank &1];
};

void	syncPRG (int AND, int OR) {
	for (int bank =0; bank <4; bank++) EMU->SetPRG_ROM8(0x8 +bank*2, GetPRGBank(bank) &AND |OR);
}

void	syncPRG_RAM (int AND, int OR) {
	for (int bank =0; bank <4; bank++) EMU->SetPRG_RAM8(0x8 +bank*2, GetPRGBank(bank) &AND |OR);
}

void	syncPRG_NROM (int AND, int OR, int bankAND, int cpuMask) {
	for (int bank =0; bank <4; bank++) EMU->SetPRG_ROM8(0x8 +bank*2, GetPRGBank(bank &bankAND) &AND &~cpuMask|OR |bank &cpuMask);
}

void	syncPRG_GNROM_66 (int A14, int AND, int OR) {
	EMU->SetPRG_ROM8(0x8, (GetPRGBank(0) &~A14 &~1) &AND |OR);
	EMU->SetPRG_ROM8(0xA, (GetPRGBank(0) &~A14 | 1) &AND |OR);
	EMU->SetPRG_ROM8(0xC, (GetPRGBank(0) | A14 &~1) &AND |OR);
	EMU->SetPRG_ROM8(0xE, (GetPRGBank(0) | A14 | 1) &AND |OR);
}

void	syncPRG_GNROM_67 (int A14, int AND, int OR) {
	EMU->SetPRG_ROM8(0x8, (GetPRGBank(0) &~A14) &AND |OR);
	EMU->SetPRG_ROM8(0xA, (GetPRGBank(1) &~A14) &AND |OR);
	EMU->SetPRG_ROM8(0xC, (GetPRGBank(0) | A14) &AND |OR);
	EMU->SetPRG_ROM8(0xE, (GetPRGBank(1) | A14) &AND |OR);
}

void	syncPRG_2 (int AND8, int ANDC, int OR8, int ORC) {
	EMU->SetPRG_ROM8(0x8, GetPRGBank(0) &AND8 |OR8);
	EMU->SetPRG_ROM8(0xA, GetPRGBank(1) &AND8 |OR8);
	EMU->SetPRG_ROM8(0xC, GetPRGBank(2) &ANDC |ORC);
	EMU->SetPRG_ROM8(0xE, GetPRGBank(3) &ANDC |ORC);
}

void	syncPRG_4 (int AND8, int ANDA, int ANDC, int ANDE, int OR8, int ORA, int ORC, int ORE) {
	EMU->SetPRG_ROM8(0x8, GetPRGBank(0) &AND8 |OR8);
	EMU->SetPRG_ROM8(0xA, GetPRGBank(1) &ANDA |ORA);
	EMU->SetPRG_ROM8(0xC, GetPRGBank(2) &ANDC |ORC);
	EMU->SetPRG_ROM8(0xE, GetPRGBank(3) &ANDE |ORE);
}

int	getCHRBank (int bank) {
	if (pointer &0x80) bank ^=4;
	if (bank &4)
		return reg[bank -2];
	else
		return reg[bank >>1] &~1 | bank&1;
};

void	syncCHR (int AND, int OR) {
	if (ROM->CHRROMSize)
		syncCHR_ROM(AND, OR);
	else
		syncCHR_RAM(AND, OR);
}

void	syncCHR_ROM (int AND, int OR) {
	for (int bank =0; bank <8; bank++) EMU->SetCHR_ROM1(bank, GetCHRBank(bank) &AND |OR);
}

void	syncCHR_ROM (int AND, int OR0, int OR4) {
	for (int bank =0; bank <4; bank++) EMU->SetCHR_ROM1(bank, GetCHRBank(bank) &AND |OR0);
	for (int bank =4; bank <8; bank++) EMU->SetCHR_ROM1(bank, GetCHRBank(bank) &AND |OR4);
}

void	syncCHR_ROM (int AND, int OR0, int OR2, int OR4, int OR6) {
	for (int bank =0; bank <2; bank++) EMU->SetCHR_ROM1(bank, GetCHRBank(bank) &AND |OR0);
	for (int bank =2; bank <4; bank++) EMU->SetCHR_ROM1(bank, GetCHRBank(bank) &AND |OR2);
	for (int bank =4; bank <6; bank++) EMU->SetCHR_ROM1(bank, GetCHRBank(bank) &AND |OR4);
	for (int bank =6; bank <8; bank++) EMU->SetCHR_ROM1(bank, GetCHRBank(bank) &AND |OR6);
}

void	syncCHR_RAM (int AND, int OR) {
	for (int bank =0; bank <8; bank++) EMU->SetCHR_RAM1(bank, GetCHRBank(bank) &AND |OR);
}

void	syncMirror (void) {
	if (mirroring &1) EMU->Mirror_H(); else EMU->Mirror_V();
}

void	syncMirrorA17 (void) {
	for (int bank =0; bank <8; bank++) EMU->SetCHR_NT1(bank |8, GetCHRBank(bank) >>7);
}

int	MAPINT	trapWRAMRead_MMC3 (int bank, int addr) {
	if (wramAlwaysEnabled || wram &0x80)
		return wramReadCallback(bank, addr);
	else
		return wramRead(bank, addr);
}

void	MAPINT	trapWRAMWrite_MMC3 (int bank, int addr, int val) {
	if ((wramAlwaysEnabled || wram &0x80) && ~wram &0x40)
		wramWriteCallback(bank, addr, val);
	else
		wramWrite(bank, addr, val);
}

int	MAPINT	trapWRAMRead_MMC6 (int bank, int addr) {
	// "If neither bank is enabled for reading, the $7000-$7FFF area is open bus.
	//  If only one bank is enabled for reading, the other reads back as zero."
	addr &=0x3FF;
	if (~addr &0x200)	// first 512 bytes
		return wram &0x20? wramRead(0x7, addr): (wram &0x80? 0x00: *EMU->OpenBus);
	else			// second 512 bytes
		return wram &0x80? wramRead(0x7, addr): (wram &0x20? 0x00: *EMU->OpenBus);
}

void	MAPINT	trapWRAMWrite_MMC6 (int bank, int addr, int val) {
	// "The write-enable bits only have effect if that bank is enabled for reading, otherwise the bank is not writable."
	addr &=0x3FF;
	if (~addr &0x200) {	// first 512 bytes
		if (wram &0x20 && wram &0x10) wramWrite(0x7, addr, val);
	} else {		// second 512 bytes 
		if (wram &0x80 && wram &0x40) wramWrite(0x7, addr, val);
	}
}

void	MAPINT	write (int bank, int addr, int val) {
	switch(bank &~1) {
		case 0x8: writeReg(bank, addr, val); break;
		case 0xA: writeMirroringWRAM(bank, addr, val); break;
		case 0xC: writeIRQConfig(bank, addr, val); break;
		case 0xE: writeIRQEnable(bank, addr, val); break;
	}
}

void	MAPINT	writeReg (int bank, int addr, int val) {
	if (addr &1)
		reg[pointer &7] =val;
	else 
		pointer =val;
	sync();
}

void	MAPINT	writeMirroringWRAM (int bank, int addr, int val) {
	if (addr &1)
		wram =val;
	else
		mirroring =val;
	sync();
}

void	MAPINT	writeIRQConfig (int bank, int addr, int val) {
	if (addr &1) {		
		if (type ==MMC3Type::Acclaim) prescaler =7;
		counter =0;
		reload =true;
	} else
		reloadValue =val;
}

void	MAPINT	writeIRQEnable (int bank, int addr, int val) {
	enableIRQ =!!(addr &1);
	if (!enableIRQ) EMU->SetIRQ(1);
}

void	MAPINT	load (FSync _sync) {
	load (_sync, MMC3Type::Sharp);
}

void	MAPINT	load (FSync _sync, MMC3Type _type) {
	sync =_sync;
	type =_type;
	revision =type ==MMC3Type::NEC || type ==MMC3Type::MMC6? MMC3Revision::MMC3B: MMC3Revision::MMC3C;
	setBankCallback(getPRGBank, getCHRBank);
	wramAlwaysEnabled =false;
}

void	setBankCallback (int (*prg)(int), int (*chr)(int)) {
	GetPRGBank =prg;
	GetCHRBank =chr;
}
	
void	setWRAMCallback (FCPURead theWRAMReadCallback, FCPUWrite theWRAMWriteCallback) {
	wramReadCallback  =theWRAMReadCallback?  theWRAMReadCallback:  wramRead;
	wramWriteCallback =theWRAMWriteCallback? theWRAMWriteCallback: wramWrite;
	sync();
}

void	MAPINT	reset (RESET_TYPE resetType) {
	if (resetType ==RESET_HARD) {
		pointer =0x00;
		reg[0] =0x00; reg[1] =0x02;
		reg[2] =0x04; reg[3] =0x05; reg[4] =0x06; reg[5] =0x07;
		reg[6] =0x00; reg[7] =0x01;
		mirroring =type==MMC3Type::FCEUX && ~ROM->INES_Flags &1? 1: 0;
		wram =0;
		enableIRQ =false;
		reload =false;
		counter =0;
		prescaler =7;
		pa12Filter =0;
		prevAddr =0;
	}
	wramRead  =wramReadCallback  =EMU->GetCPUReadHandler(0x6);
	wramWrite =wramWriteCallback =EMU->GetCPUWriteHandler(0x6);
	for (int bank =0x6; bank<=0x7; bank++) {
		EMU->SetCPUReadHandler     (bank, type ==MMC3Type::MMC6? trapWRAMRead_MMC6: trapWRAMRead_MMC3);
		EMU->SetCPUReadHandlerDebug(bank, type ==MMC3Type::MMC6? trapWRAMRead_MMC6: trapWRAMRead_MMC3);
		EMU->SetCPUWriteHandler    (bank, type ==MMC3Type::MMC6? trapWRAMWrite_MMC6: trapWRAMWrite_MMC3);
	}
	
	for (int bank =0x8; bank<=0x9; bank++) EMU->SetCPUWriteHandler(bank, writeReg);
	for (int bank =0xA; bank<=0xB; bank++) EMU->SetCPUWriteHandler(bank, writeMirroringWRAM);
	for (int bank =0xC; bank<=0xD; bank++) EMU->SetCPUWriteHandler(bank, writeIRQConfig);
	for (int bank =0xE; bank<=0xF; bank++) EMU->SetCPUWriteHandler(bank, writeIRQEnable);
	EMU->SetIRQ(1);
	sync();
}

void	clockCounter () {
	uint8_t prevCounter =counter;
	counter =!counter? reloadValue: --counter;
	if ((prevCounter || reload || type==MMC3Type::Sharp) && !counter && enableIRQ) EMU->SetIRQ(0);
	reload =false;
}

void	MAPINT	cpuCycle (void) {
	if (type !=MMC3Type::Acclaim && pa12Filter) pa12Filter--;
}

void	MAPINT	ppuCycle (int addr, int scanline, int cycle, int) {
	bool hBlank =false;
	if (type ==MMC3Type::FCEUX)
		hBlank =scanline >=0 && scanline <239 && cycle ==256;
	else
	if (type ==MMC3Type::Acclaim) {
		if (prevAddr &0x1000 && ~addr &0x1000 && !(prescaler++ &7)) hBlank =true;
		prevAddr =addr;
	} else
	if (addr &0x1000) {
		if (!pa12Filter) hBlank =true;
		pa12Filter =3;
	}	
	if (hBlank) clockCounter();
}

int	MAPINT	saveLoad (STATE_TYPE stateMode, int offset, unsigned char *data) {
	SAVELOAD_BYTE(stateMode, offset, data, pointer);
	for (int i =0; i <8; i++) SAVELOAD_BYTE(stateMode, offset, data, reg[i]);
	SAVELOAD_BYTE(stateMode, offset, data, mirroring);
	SAVELOAD_BYTE(stateMode, offset, data, wram);
	SAVELOAD_BYTE(stateMode, offset, data, counter);
	SAVELOAD_BYTE(stateMode, offset, data, reloadValue);
	SAVELOAD_BOOL(stateMode, offset, data, reload);
	SAVELOAD_BOOL(stateMode, offset, data, enableIRQ);
	if (type ==MMC3Type::Acclaim) {
		SAVELOAD_BYTE(stateMode, offset, data, prescaler);
		SAVELOAD_WORD(stateMode, offset, data, prevAddr);
	} else
		SAVELOAD_BYTE(stateMode, offset, data, pa12Filter);
	if (stateMode ==STATE_LOAD) sync();
	return offset;
}
} // namespace MMC3
